/************************************************************************/
/* File name     : xpatlex.c                                              */
/************************************************************************/
/* Written by    : Robert Siegmund, BID PL24 R&D, 06/03/97              */
/* Modified by   : Stefan Friebe, BID PL24 R&D, 07/Oct/97        */
/* Modified by   : Stefan Friebe, BVS CVT R&D, 21/Jun/00        */
/* Last modified : ask clearcase                                        */
/************************************************************************/
/* File contents :                                                      */
/*                patternterm string condition scanner and parser call    */
/*                                    */
/************************************************************************/
/* NOTES :                                                              */
/************************************************************************/

#include <stdio.h>
#include <string.h>
#include "xstreq.h"
#include "xtypedef.h"
#include "xpatlex.h"
#include "xpattdef.h"
#include <xerrcapi.h>
#include "xutil.h"  /* BESTX Defines for string functions */
#include <xaddrmap.h>  /* Addresses of DBI registers */
#include <xsession.h>  /* DirectRegWrite function */



/* xpatytok.h is automatically generated by yacc -d  */
/* and moved to the capi/include directory       */
/* for further information see the make file     */

#include "xpatytok.h"


bx_int8  err_flag;           /* this is the global error flag */

char parser_errstring[64]; 
static char* cond_buffer;      /* pointer to the pattern term string to parse */


/* global array that holds the c and dc pattern for E2926A    */
/* first it is initialized to all dc */
bx_int32 c_pattern[BX_PATTARR_SIZE];   /* patterns terms HW representation */
bx_int32 dc_pattern[BX_PATTARR_SIZE];  /* mask for pattern terms HW representation */

/* global variable that indicates whether the pattern term is always zero */
bx_bool  global_zero;

static bx_patt_infotype *e29_patt_signal_table;

extern bx_patt_infotype e2929a_patt_signal_table[];
extern bx_patt_infotype e2930a_patt_signal_table[];

extern int     yy_patt_char;       /* yacc yyparse() last read token      */
static int     yybrflag;           /* context switch flag for '0' and '1' */
extern long    yy_patt_maxdepth;   /* max stack size of yacc parser       */

bx_int16 bufptr;                   /* global pointer to next char in cond_buffer*/
bx_int8 yy_patt_parse_dest;        /* what the parser output is for             */

/******************************************************************************/
/* yy_patt_error(), called by yyparse() when yacc finds syntax errors         */
/******************************************************************************/

int yy_patt_error(char * s)
{
  if(err_flag == 0)  /* when error flag is set, message is already      */
                    /* written to error string buffer                  */
  {
    if(yy_patt_char == 256)  /* YYERRCODE */
        BESTX_STRCPY(parser_errstring,s);
    else if(yy_patt_char == 0)
        BESTX_STRCPY(parser_errstring,": Improperly terminated equation");
    else if(yy_patt_char >= 0x1f && yy_patt_char <= 0xff)
        sprintf(parser_errstring," near '%c'",yy_patt_char);
    else if(yy_patt_char == UNDEF_TOKEN)
        BESTX_STRCPY(parser_errstring,": Unexpected 'T()'");

    err_flag = SYNTAX_ERROR;
  }
  DBG_OUT(("Errorflag : %d\n",err_flag));
  DBG_OUT(("Errorstring : %s\n",parser_errstring));
  return (0);
}


/************************************************************************/
/*   Function parse_eqn()                                               */
/*                                                                      */
/*   Parses the pattern term from CAPI and optionally programs HW       */
/*                                                                      */
/*   Parameters: char* patternterm    - string with equation to parse   */
/*         bx_patttype        - Type of pattern term (Obs,bus,err,tran) */
/*         bx_bool            - wether to write to HW or not            */
/*                                                                      */
/*   Called from BestXPattSet() and BestXPattProg()                     */
/*                                                                      */
/************************************************************************/

bx_errtype parse_eqn(bx_handletype handle, bx_charptrtype patternterm,
                     bx_patttype dest, bx_bool write_to_hw)
{
  bx_errtype yacc_errstatus;
  bx_int8 i;
  bx_hwtype theBoard;
  bx_int32 reg_offset = 0;
  bx_int32 temp_swap = 0;
  bx_int32 ObsTermAddressDiff,BusTermAddressDiff;      

  BX_TRY_VARS_NO_PROG;

  global_zero = BX_FALSE;        /* initialize global_zero flag */
  yybrflag = 0;                  /* context switch flag for '0' and '1' */

  /* Find out, which type of pattern to parse */
  if ((dest >= BX_PATT_BUS0) && (dest <= BX_PATT_BUS3)) /* this includes the two COND patterns !! */
  { 
    yy_patt_parse_dest = BUS_PATT;
  } 
  else if ((dest >= BX_PATT_OBS0) && (dest <= BX_PATT_OBS5))
  {
    yy_patt_parse_dest = OBS_PATT;
  }
  else if ((dest >= BX_PATT_ERR0) && (dest <= BX_PATT_ERR0))
  {
    yy_patt_parse_dest = ERR_PATT;
  }
  else if ((dest >= BX_PATT_TRAN0) && (dest <= BX_PATT_TRAN0))
  {
    yy_patt_parse_dest = TRAN_PATT;
  }
  else
  {
    /* range check already done by caller; we should never get here ! */
    BX_E_ERROR_MSG_SET("parse_eqn: Invalid pattern term type");
    return (BX_E_ERROR);
  }

  yy_patt_maxdepth = YYMAXDEPTH;
  
  /* initialize pattern terms to logic '1' (all inputs are don't care) */
  for(i = 0; i < BX_PATTARR_SIZE; i++)
  {
    c_pattern[i] = 0x00000000;  /* Pattern: trigger, if signal is low (mask must be 0) */
    dc_pattern[i] = 0xFFFFFFFF; /* Mask: disable all */
  } 

  /* set global pointer to next char in cond_buffer*/
  bufptr = 0;

  /* set cond_buffer pointer to the location of patternterm */
  cond_buffer = patternterm;

  yacc_errstatus = BX_E_OK;

  /* chris: 
     Because there is no handle available in function 
     yy_patt_parse() (which calls yy_patt_lex()) 
     we init the global variable e29_patt_signal_table here
     and then use it in yy_patt_lex().
  */
  BestXGetHWFromHandle(handle, &theBoard);
  switch(theBoard)
  {
    case BX_HW_E2929A:
    case BX_HW_E2929A_DEEP:
    case BX_HW_E2929B:
    case BX_HW_E2929B_DEEP:
    case BX_HW_E2922A:
    case BX_HW_E2922B:
      e29_patt_signal_table=e2929a_patt_signal_table;
      break;
    
    case BX_HW_E2930A:
    case BX_HW_E2930B:
    case BX_HW_E2923A:
    case BX_HW_E2923B:
    case BX_HW_E2923F:
      if (BestXIsMode2(handle))
      {
        e29_patt_signal_table=e2930a_patt_signal_table;
      }
      else
      {
        e29_patt_signal_table=e2929a_patt_signal_table;
      }

      break;
    default: 
      yacc_errstatus=BX_E_INVALID_CASE;
  } /*switch*/

  err_flag = 0;  /* clear global parserd error flag */

  /* Parse the equation located in cond_buffer 
     and fill dc_pattern[] and c_pattern */
  yy_patt_parse(); /* no HW access here ! */

  if (err_flag != 0 )
  {
    /* Parse error occurred */

    /* Pattern type is returned in BX_ERRPAR_1 */
    /* This info is used to retrieve the pattern string from the database */
    BestXLastErrorParamSet(handle, BX_ERRPAR_1,(bx_int32) dest);

    /* Convert yacc errors to CAPI errors */
    switch (err_flag)
    {
      case UNDEFINED_TOKEN:
        yacc_errstatus = BX_E_PATT_UNDEF_TOKEN;
        BestXLastErrorParamSet(handle, BX_ERRPAR_2,bufptr);
        break;
      case SYNTAX_ERROR:
        yacc_errstatus = BX_E_PATT_SYNTAX;
        BestXLastErrorParamSet(handle, BX_ERRPAR_2,bufptr);
        break;
      case WRONG_PARSEDEST_DPT:
      case WRONG_PARSEDEST_SPT:
        yacc_errstatus = BX_E_PATT_SYNTAX;
        BestXLastErrorParamSet(handle, BX_ERRPAR_2,bufptr);
        break;
      case MORE_THAN_ONE_LISTSIGNAL:
        yacc_errstatus = BX_E_PATT_MTO_LISTSIGNAL;
        BestXLastErrorParamSet(handle, BX_ERRPAR_2,bufptr);
        break;
      default:
        yacc_errstatus = BX_E_PATT_UNKNOWN_ERR;
        break;    
    }
    DBG_OUT(("Error Flag is now : %d\n",err_flag));
  }
  
  if (yacc_errstatus == BX_E_OK)
  {
    /* No error occurred */

    for(i = 0; i < BX_PATTARR_SIZE; i++)
    {
      DBG_OUT(("c_pattern[%d]=0x%08X dc_pattern[%d]=0x%08X \n",i,c_pattern[i],i,dc_pattern[i]));
    }

    if (write_to_hw)
    {
      /* Because of additional AD lines in Mode2, Faust based cards may have a different address map layout. 
         Compute address difference between two blocks of the same pattern term type. */
      if (BestXHasMephisto(handle))
      {
        ObsTermAddressDiff=0x40;
        BusTermAddressDiff=0x40;
      }
      else
      {
        ObsTermAddressDiff=0x40;
        BusTermAddressDiff=0x40;
      }

      BX_TRY_BEGIN
      {
        /* Turn off pattern term (output enables) */
        switch (yy_patt_parse_dest)
        {
          case TRAN_PATT:
            BX_TRY(BestXDirectRegWrite(handle,(BestXHasMephisto(handle)?BX_REG_TRAN_OE_REG_M:BX_REG_TRAN_OE_REG_F),sizeof(bx_int16),1));
            break;      
          case OBS_PATT:     
            reg_offset = dest - BX_PATT_OBS0;
            BX_TRY(BestXDirectRegWrite(handle,((BestXHasMephisto(handle)?BX_REG_BLK0_OBS_OE_REG_M:BX_REG_BLK0_OBS_OE_REG_F) + (reg_offset * ObsTermAddressDiff)),sizeof(bx_int16),1));
            break;
          case ERR_PATT:
            BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_OE_REG,sizeof(bx_int16),1));
            break;
          case BUS_PATT: /* bus/cond pattern */
            reg_offset = dest - BX_PATT_BUS0;
            BX_TRY(BestXDirectRegWrite(handle,((BestXHasMephisto(handle)?BX_REG_BLK0_BUS_OE_REG_M:BX_REG_BLK0_BUS_OE_REG_F) + (reg_offset * BusTermAddressDiff)),sizeof(bx_int16),1));
            break;
          default:
            /* We should never get here */
            return BX_E_ERROR;
            break;
        }

        if (global_zero)
        {
          /* keep pattern term turned off */
          DBG_OUT(("Pattern Term is always zero!\n"));
        }
        else
        {
          /* program pattern term registers and enable the outputs again */

          if ((dest >= BX_PATT_BUS0) && (dest <= BX_PATT_COND1))
          {
            reg_offset = dest - BX_PATT_BUS0;

            /* swap upper and lower 16 bits in dc an c_pattern[0,1] due to 
              strange hardware register layout */
            
            temp_swap = c_pattern[0] & 0xFFFF;
            c_pattern[0] = c_pattern[0] >> 16;
            c_pattern[0] = c_pattern[0] & 0xFFFF | ((temp_swap << 16) & 0xFFFF0000);
            temp_swap = dc_pattern[0] & 0xFFFF;
            dc_pattern[0] = dc_pattern[0] >> 16;
            dc_pattern[0] = dc_pattern[0] & 0xFFFF | ((temp_swap << 16) & 0xFFFF0000);
            temp_swap = c_pattern[1] & 0xFFFF;
            c_pattern[1] = c_pattern[1] >> 16;
            c_pattern[1] = c_pattern[1] & 0xFFFF | ((temp_swap << 16) & 0xFFFF0000);
            temp_swap = dc_pattern[1] & 0xFFFF;
            dc_pattern[1] = dc_pattern[1] >> 16;
            dc_pattern[1] = dc_pattern[1] & 0xFFFF | ((temp_swap << 16) & 0xFFFF0000);

            if (BestXHasMephisto(handle))
            {
              for (i = 0; i < BX_PATTARR_SIZE; i++)
              {
                BX_TRY(BestXDirectRegWrite(handle,(BX_REG_BLK0_BUS_PATTERN_0_M+(reg_offset * BusTermAddressDiff)+i*4),sizeof(bx_int32),c_pattern[i]));
                BX_TRY(BestXDirectRegWrite(handle,(BX_REG_BLK0_MASK_BUS_PATTERN_0_M+(reg_offset * BusTermAddressDiff)+i*4),sizeof(bx_int32),dc_pattern[i]));
              }
              BX_TRY(BestXDirectRegWrite(handle,(BX_REG_BLK0_BUS_OE_REG_M + (reg_offset * BusTermAddressDiff)),sizeof(bx_int16),3));
            }
            else
            {
              /* Faust based cards */
              for (i = 0; i < BX_PATTARR_SIZE; i++)
              {
                BX_TRY(BestXDirectRegWrite(handle,(BX_REG_BLK0_BUS_PATTERN_0_F+(reg_offset * BusTermAddressDiff)+i*4),sizeof(bx_int32),c_pattern[i]));
                BX_TRY(BestXDirectRegWrite(handle,(BX_REG_BLK0_MASK_BUS_PATTERN_0_F+(reg_offset * BusTermAddressDiff)+i*4),sizeof(bx_int32),dc_pattern[i]));
              }
              BX_TRY(BestXDirectRegWrite(handle,(BX_REG_BLK0_BUS_OE_REG_F + (reg_offset * BusTermAddressDiff)),sizeof(bx_int16),3));
            }
          } 
          else if ((dest >= BX_PATT_OBS0) && (dest <= BX_PATT_OBS5))
          {
            reg_offset = dest - BX_PATT_OBS0;
           
            if (BestXHasMephisto(handle))
            {
              for (i = 0; i < BX_PATTARR_SIZE; i++)
              {
                BX_TRY(BestXDirectRegWrite(handle,(BX_REG_BLK0_OBS_PATTERN_0_M+(reg_offset * ObsTermAddressDiff)+i*4),sizeof(bx_int32),c_pattern[i]));
                BX_TRY(BestXDirectRegWrite(handle,(BX_REG_BLK0_MASK_OBS_PATTERN_0_M+(reg_offset * ObsTermAddressDiff)+i*4),sizeof(bx_int32),dc_pattern[i]));
              }
              BestXDirectRegWrite(handle,(BX_REG_BLK0_OBS_OE_REG_M + (reg_offset * ObsTermAddressDiff)),sizeof(bx_int16),3);
            }
            else
            {
              for (i = 0; i < BX_PATTARR_SIZE; i++)
              {
                BX_TRY(BestXDirectRegWrite(handle,(BX_REG_BLK0_OBS_PATTERN_0_F+(reg_offset * ObsTermAddressDiff)+i*4),sizeof(bx_int32),c_pattern[i]));
                BX_TRY(BestXDirectRegWrite(handle,(BX_REG_BLK0_MASK_OBS_PATTERN_0_F+(reg_offset * ObsTermAddressDiff)+i*4),sizeof(bx_int32),dc_pattern[i]));
              }
              BestXDirectRegWrite(handle,(BX_REG_BLK0_OBS_OE_REG_F + (reg_offset * ObsTermAddressDiff)),sizeof(bx_int16),3);
            }
          } 
          else if ((dest >= BX_PATT_ERR0) && (dest <= BX_PATT_ERR0))
          { 
            BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_PATTERN,sizeof(bx_int32),c_pattern[0]));
            BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_MASK_PATTERN,sizeof(bx_int32),dc_pattern[0]));
            BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_OE_REG,sizeof(bx_int32),3));
          }
          else 
          {
            /* Transitional Pattern Term */
            if (BestXHasMephisto(handle))
            {
              /* Swap upper and lower 16 bits in dc an c_pattern[0,1] due to strange hardware register layout */
              temp_swap = c_pattern[0] & 0xFFFF;
              c_pattern[0] = c_pattern[0] >> 16;
              c_pattern[0] = c_pattern[0] & 0xFFFF | ((temp_swap << 16) & 0xFFFF0000);
              temp_swap = dc_pattern[0] & 0xFFFF;
              dc_pattern[0] = dc_pattern[0] >> 16;
              dc_pattern[0] = dc_pattern[0] & 0xFFFF | ((temp_swap << 16) & 0xFFFF0000);
              temp_swap = c_pattern[1] & 0xFFFF;
              c_pattern[1] = c_pattern[1] >> 16;
              c_pattern[1] = c_pattern[1] & 0xFFFF | ((temp_swap << 16) & 0xFFFF0000);
              temp_swap = dc_pattern[1] & 0xFFFF;
              dc_pattern[1] = dc_pattern[1] >> 16;
              dc_pattern[1] = dc_pattern[1] & 0xFFFF | ((temp_swap << 16) & 0xFFFF0000);

              for (i = 0; i < BX_PATTARR_SIZE-1; i++) 
              /* -1, because signal address_phase (the only used bit in this (Mephisto-) register) is not needed (generated from FRAME#) */
              {
                BX_TRY(BestXDirectRegWrite(handle,(BX_REG_TRAN_PAT_MASK_0+i*4),sizeof(bx_int32),dc_pattern[i]));
              }
              BX_TRY(BestXDirectRegWrite(handle,BX_REG_TRAN_OE_REG_M,sizeof(bx_int16),3));
            }
            else
            {
              for (i = 0; i < BX_PATTARR_SIZE-1; i++) 
              /* -1, because signal address_phase (the only used bit in this (Mephisto-) register) is not needed (generated from FRAME#).
                Faust based cards implement some other bits (ECC) here, so we may need a change here in the future. */
              {
                BX_TRY(BestXDirectRegWrite(handle,(BX_REG_TRAN_PAT_MASK_0+i*4),sizeof(bx_int32),dc_pattern[i]));
              }
              BX_TRY(BestXDirectRegWrite(handle,BX_REG_TRAN_OE_REG_F,sizeof(bx_int16),3));
            }
          }
        }
      }   /* BX_TRY_BEGIN */
    }
  }
  BX_ERRETURN (yacc_errstatus);
}


/************************************************************************/
/* Lexicographic analysis of pattern term equations.                    */
/* This function is called by yy_patt_parse() and returns the next found*/
/* token to the yacc-generated syntactic analyzer.                      */
/************************************************************************/

int yy_patt_lex()
{
  int i = 0;
  int list_flag;
  char c;
  char token[48];
  bx_int8  yylex_num_error = 0;
  bx_int32 yynumval,yynumdc;



  /* scan buffer for tokens and pass them to the yacc parser */

  do {

       if (err_flag != 0)
       {
     return(UNDEF_TOKEN);
       }
    
       c = cond_buffer[bufptr++];

       if(is_ascii(c))
     token[i++] = c;
       else
       {
     token[i] = '\0';  /* terminate string */
     
     /* check if token is the 'toggle' function identifier */
     
     if((c == '(') && (!BESTX_STRCASECMP(token,"T")))
     {
       bufptr--;
       if(yy_patt_parse_dest == TRAN_PATT)
         return(TOGGLE);
       else
         err_flag = UNDEFINED_TOKEN;
         return(UNDEF_TOKEN);
     }
     
     /* check if token is the 'TOGGLE_ALL' alias */
     
     if(!BESTX_STRCASECMP(token,"ALL"))
     {
       bufptr--;
       if(yy_patt_parse_dest == TRAN_PATT)
         return(TOGGLE_ALL);
       else
         err_flag = UNDEFINED_TOKEN;
         return(UNDEF_TOKEN);
     }

     
     /* check if token is the all zero '0' or all one '1'          */
     
     if(!BESTX_STRCMP(token,"0"))
     {
       if (c != '\\' && yybrflag == 0)
       {
         bufptr--; return(CONST_0);
       }
       else if (c == '\\' && cond_buffer[bufptr] == 'b')
       {
         bufptr++; return(CONST_0);
       }
     }

     if(!BESTX_STRCMP(token,"1"))
     {
       if (c != '\\' && yybrflag == 0)
       {
         bufptr--; return(CONST_1);
       }
       else if (c == '\\' && cond_buffer[bufptr] == 'b')
       {
         bufptr++; return(CONST_1);
       }
     }

     if(!BESTX_STRCASECMP(token,"X"))
     {
       if (c != '\\' && yybrflag == 0)
       {
         bufptr--; return(DONT_CARE);
       }
       else if (c == '\\' && cond_buffer[bufptr] == 'b')
       {
         bufptr++; return(DONT_CARE);
       }
     }
     
     /* check if token is any number except 0 and 1 */
     
     yylex_num_error = 0;
     
     /* check if token is a decimal number */
     
     if(!((c == '\\' && cond_buffer[bufptr] == 'b') ||
          (c == '\\' && cond_buffer[bufptr] == 'h')) && BESTX_STRLEN(token))
     {
       for(i = 0;i < (int)BESTX_STRLEN(token);i++)
         if(token[i] < '0' || token[i] > '9')
           yylex_num_error = 1;
       if(!yylex_num_error)
       {
         if(c == '\\' && cond_buffer[bufptr] == 'd')
           bufptr++;
         else
           bufptr--;
         conv_string_to_int(token,10,&yynumval,&yynumdc);
         yy_patt_lval.int_value = yynumval;
         /* Decimal number detected */
         return(DEC_NUMBER);
       }
     }
     
         /* check if token is a hex number */
     
     else if(c == '\\' && cond_buffer[bufptr] == 'h' && BESTX_STRLEN(token))
     {
       for(i = 0;i < (int)BESTX_STRLEN(token);i++)
         if(!(token[i] >= '0' && token[i] <= '9' ||
          token[i] >= 'a' && token[i] <= 'f' ||
          token[i] >= 'A' && token[i] <= 'F' ||
          token[i] == 'X'   || token[i] == 'x'))
           yylex_num_error = 1;
       if(!yylex_num_error)
       {
         bufptr++;
         conv_string_to_int(token,16,&yynumval,&yynumdc);
         yy_patt_lval.num_value.c_bits  = yynumval;
         yy_patt_lval.num_value.dc_bits = yynumdc;
         return(HEX_NUMBER);
       }
     }
     
     /* check if token is a bin number */
     
     else if(c == '\\' && cond_buffer[bufptr] == 'b' && BESTX_STRLEN(token))
     {
       for(i = 0;i < (int)BESTX_STRLEN(token);i++)
         if(token[i] != '0' && token[i] != '1' &&
        token[i] != 'X' && token[i] != 'x')
           yylex_num_error = 1;
       if(!yylex_num_error)
       {
         bufptr++;
         conv_string_to_int(token,2,&yynumval,&yynumdc);
         yy_patt_lval.num_value.c_bits  = yynumval;
         yy_patt_lval.num_value.dc_bits = yynumdc;
         return(BIN_NUMBER);
       }
     }
     
     /* check if token is a valid signal name */
     
     if(BESTX_STRLEN(token))
     {
       i = 0;
       while( e29_patt_signal_table[i].Name != NULL)
       {
         if(!BESTX_STRCASECMP(token,e29_patt_signal_table[i].Name))
         {
           list_flag = BX_FALSE;
           
           /* initialize values */
           yy_patt_lval.signal_idx.C_value = 0x0;
           yy_patt_lval.signal_idx.DC_value = 0x0;           
           
           if(yy_patt_parse_dest == BUS_PATT)
           {
             if (e29_patt_signal_table[i].MemPosBus == NO_BIT_POS)
             {
               break;
             }
         
             yy_patt_lval.signal_idx.Position = (bx_int8)(e29_patt_signal_table[i].MemPosBus & 0x1F);
             yy_patt_lval.signal_idx.Index    = (bx_int8)((e29_patt_signal_table[i].MemPosBus >> 5) & 0x7);
                           
           }
           else if (yy_patt_parse_dest == OBS_PATT)
           {
             if (e29_patt_signal_table[i].MemPosObs == NO_BIT_POS)
             {
               break;
             }
             yy_patt_lval.signal_idx.Position = (bx_int8)(e29_patt_signal_table[i].MemPosObs & 0x1F);
             yy_patt_lval.signal_idx.Index    = (bx_int8)((e29_patt_signal_table[i].MemPosObs >> 5) & 0x7);
         
             if((e29_patt_signal_table[i].MemPosObs & 0x100) != 0)
             {
               list_flag = BX_TRUE;
             }
           }
           else if (yy_patt_parse_dest == ERR_PATT)
           {
             if (e29_patt_signal_table[i].MemPosErr == NO_BIT_POS)
             {
               break;
             }
             yy_patt_lval.signal_idx.Position = (bx_int8)(e29_patt_signal_table[i].MemPosErr & 0x1F);
             yy_patt_lval.signal_idx.Index    = (bx_int8)((e29_patt_signal_table[i].MemPosErr >> 5) & 0x7);
         
           }
           else if (yy_patt_parse_dest == TRAN_PATT)
           {
             if (e29_patt_signal_table[i].MemPosTran == NO_BIT_POS)
             {
                break;
             }
             yy_patt_lval.signal_idx.Position = (bx_int8)(e29_patt_signal_table[i].MemPosTran & 0x1F);
             yy_patt_lval.signal_idx.Index    = (bx_int8)((e29_patt_signal_table[i].MemPosTran >> 5) & 0x7);
         
           }
           
           bufptr--;   /* put back last character */
           
           
           /* check for type */
           if(e29_patt_signal_table[i].BitCount > 1)
           {
             if(list_flag == BX_TRUE)
             {
               yy_patt_lval.signal_idx.Type = LISTTYPE;
               yy_patt_lval.signal_idx.Length = 0x1 << ((bx_int8) (e29_patt_signal_table[i].BitCount & 0xFF));
               return(SIGNAL_LIST);
             }
             else
             {
               yy_patt_lval.signal_idx.Type = VECTORTYPE;
               yy_patt_lval.signal_idx.Length = (bx_int8) (e29_patt_signal_table[i].BitCount & 0xFF);
               return(SIGNAL_VECTOR);
             }
           }
           else if (yy_patt_parse_dest == ERR_PATT)
           {
              yy_patt_lval.signal_idx.Type = BITTYPE;
              yy_patt_lval.signal_idx.Length = (bx_int8) (e29_patt_signal_table[i].BitCount & 0xFF);
              return(ERR_SIGNAL);
           }
           else
           {
              yy_patt_lval.signal_idx.Type = BITTYPE;
              yy_patt_lval.signal_idx.Length = (bx_int8) (e29_patt_signal_table[i].BitCount & 0xFF);
              return(SIGNAL);
           }
         }
         i++;
       }
       /* signal 'token' was not found in list */
       BESTX_STRCPY(parser_errstring,token);
       err_flag = UNDEFINED_TOKEN;
       return(UNDEF_TOKEN);
     }
     
     
     switch(c)
     {
      case '(': case ')':
      case ':': case ',':
      case '^': case '!':
      case '<': case '>':
       return(c);
       
      case '[': yybrflag = 1;
       return(c);
       
      case ']': yybrflag = 0;
       return(c);
       
      case '|': case '&':
      {
        /* ATTENTION, Two and one | or & are accepted */
        if(cond_buffer[bufptr] == c) bufptr++;
        return(c);
      }
      
      case '=':
       if(cond_buffer[bufptr] == c)
       {
         bufptr++;
       } /* ATTENTION, two and one '=' accepted */
       return(EQU);
       /*else*//* break; */
       
      default: break;
       
     }
       }
  } while(c != '\0');
  
  return(0);
  
}


/*  own is_ascii() check. Did not use is_ascii() of  */
/*  stdlib since I want to be compatible to psos.    */

int is_ascii(char c)
{
  if((c >= '0' && c <= '9') || (c >= 'A' && c <= 'Z') ||
     (c >= 'a' && c <= 'z') || (c == '_'))
          return 1;
  else
          return 0;
}


/*************************************************************************/
/*  Function conv_string_to_int                                          */
/*  convert a string with number characters and dont_cares to a number   */
/*************************************************************************/

void conv_string_to_int(char *number,int base,bx_int32 *number_val,bx_int32 *number_dc)
{
  size_t len;
  bx_int32 j;
  bx_int32 num_val=0,dont_care=0;
  bx_int32 powers_of_10[10]={1,10,100,1000,10000,100000,1000000,10000000,100000000,1000000000};

 len=BESTX_STRLEN(number);

 /* convert to decimal number */

 if(base==10)
  {
   for(j=0;j < len;j++)
     switch(number[len-j-1]) {
         case '0': break;
         case '1': num_val+=powers_of_10[j];   break;
         case '2': num_val+=powers_of_10[j]*2; break;
         case '3': num_val+=powers_of_10[j]*3; break;
         case '4': num_val+=powers_of_10[j]*4; break;
         case '5': num_val+=powers_of_10[j]*5; break;
         case '6': num_val+=powers_of_10[j]*6; break;
         case '7': num_val+=powers_of_10[j]*7; break;
         case '8': num_val+=powers_of_10[j]*8; break;
         case '9': num_val+=powers_of_10[j]*9; break;
        }
  }

 /* convert to hexadecimal number */

 else if(base==16)
  {
   for(j=0;j < len;j++)
     switch(number[len-j-1]) {
         case '0': break;
         case '1': num_val+=(0x1 << (4*j)); break;
         case '2': num_val+=(0x2 << (4*j)); break;
         case '3': num_val+=(0x3 << (4*j)); break;
         case '4': num_val+=(0x4 << (4*j)); break;
         case '5': num_val+=(0x5 << (4*j)); break;
         case '6': num_val+=(0x6 << (4*j)); break;
         case '7': num_val+=(0x7 << (4*j)); break;
         case '8': num_val+=(0x8 << (4*j)); break;
         case '9': num_val+=(0x9 << (4*j)); break;
         case 'a':
         case 'A': num_val+=(0xA << (4*j)); break;
         case 'b':
         case 'B': num_val+=(0xB << (4*j)); break;
         case 'c':
         case 'C': num_val+=(0xC << (4*j)); break;
         case 'd':
         case 'D': num_val+=(0xD << (4*j)); break;
         case 'e':
         case 'E': num_val+=(0xE << (4*j)); break;
         case 'f':
         case 'F': num_val+=(0xF << (4*j)); break;
         case 'x':
         case 'X': dont_care+=(0xF << (4*j)); break;
      }
  }

 /* convert to binary number */

 else if(base==2)
  {
   for(j=0;j < len;j++)
     switch(number[len-j-1]) {
         case '0': break;
         case '1': num_val+=(0x1 << j); break;
         case 'x':
         case 'X': dont_care+=(0x1 << j); break;
     }
   }

  *number_val=num_val;
  *number_dc=dont_care;

}
